<?php

class AnticWP_Admin {
    private $license;

    public function __construct( $license ) {
        $this->license = $license;
        add_action( 'admin_menu', array( $this, 'add_menu' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_styles' ) );
    }

    public function enqueue_styles() {
        wp_enqueue_style( 'anticwp-admin-style', ANTICWP_URL . 'assets/css/admin-style.css', array(), ANTICWP_VERSION );
    }

    public function add_menu() {
        add_submenu_page( 'woocommerce', __( 'AnticWP Settings', 'anticwp' ), __( 'AnticWP Payments', 'anticwp' ), 'manage_woocommerce', 'anticwp-settings', array( $this, 'render_page' ) );
    }

    public function register_settings() {
        register_setting( 'anticwp_group', 'anticwp_shipping_cost' );
        register_setting( 'anticwp_group', 'anticwp_cod_cost' );
        register_setting( 'anticwp_group', 'anticwp_shipping_label' );
        register_setting( 'anticwp_group', 'anticwp_cod_label' );
        register_setting( 'anticwp_group', 'anticwp_is_enabled' );
        register_setting( 'anticwp_group', 'anticwp_cache_compatible' ); 

        if ( isset( $_POST['anticwp_activate_license'] ) ) {
            check_admin_referer( 'anticwp_license_action' );
            $key = sanitize_text_field( $_POST['license_key'] );
            if ( $this->license->activate( $key ) ) {
                add_settings_error( 'anticwp', 'active', __( 'Plugin Activated!', 'anticwp' ), 'success' );
            } else {
                add_settings_error( 'anticwp', 'error', __( 'Invalid Key.', 'anticwp' ), 'error' );
            }
        }
    }

    public function render_page() {
        $this->license->check_license_status();
        $is_active = $this->license->is_valid();
        
        $expiry_date = get_option( 'anticwp_license_expiry' );
        $days_left = __( 'UNLIMITED', 'anticwp' );
        $expiry_display = __( 'Unlimited (2099)', 'anticwp' );
        $is_unlimited = false;
        
        if ( $is_active && ! empty( $expiry_date ) ) {
            $d_start = new DateTime( 'now' );
            $d_end   = new DateTime( $expiry_date );
            $year = $d_end->format('Y');
            
            if ($year > 2090) {
                $is_unlimited = true;
                $days_left = __( 'UNLIMITED', 'anticwp' );
                $expiry_display = __( 'Unlimited', 'anticwp' );
            } else {
                if ( $d_start < $d_end ) {
                    $days_left = $d_start->diff( $d_end )->days;
                    $expiry_display = date_i18n( get_option( 'date_format' ), strtotime( $expiry_date ) );
                } else {
                    $days_left = 0;
                    $expiry_display = __( 'Expired', 'anticwp' );
                }
            }
        }
        
        $enabled = get_option('anticwp_is_enabled', 'yes');
        $cache_compatible = get_option('anticwp_cache_compatible', 'no');
        ?>

        <div class="wrap anticwp-wrap">
            <div class="anticwp-hero">
                <div class="hero-content">
                    <h1>
                        <div class="hero-icon"><span class="dashicons dashicons-tickets-alt"></span></div>
                        AnticWP
                    </h1>
                    <p><?php _e( 'Split Payment Gateway for WooCommerce', 'anticwp' ); ?></p>
                </div>
                <div class="version-badge">
                    <span class="version-label"><?php _e( 'Version', 'anticwp' ); ?></span>
                    <?php echo ANTICWP_VERSION; ?>
                </div>
            </div>

            <div class="anticwp-grid">
                <div class="main-column">
                    <form method="post" action="options.php">
                        <?php settings_fields( 'anticwp_group' ); ?>
                        
                        <div class="anticwp-card">
                            <div class="card-header">
                                <h2><span class="dashicons dashicons-cart header-icon"></span> <?php _e( 'Billing Settings', 'anticwp' ); ?></h2>
                            </div>
                            
                            <div class="card-body">
                                <div class="input-row">
                                    <label class="input-label"><?php _e( 'Plugin Mode', 'anticwp' ); ?></label>
                                    <select name="anticwp_is_enabled" class="anticwp-input" style="max-width: 200px;">
                                        <option value="yes" <?php selected($enabled, 'yes'); ?>><?php _e( 'Active', 'anticwp' ); ?></option>
                                        <option value="no" <?php selected($enabled, 'no'); ?>><?php _e( 'Inactive', 'anticwp' ); ?></option>
                                    </select>
                                </div>

                                <div class="divider"></div>

                                <div class="section-block">
                                    <h3 class="section-title">
                                        <div class="icon-box blue"><span class="dashicons dashicons-truck"></span></div>
                                        <?php _e( 'Shipping Costs', 'anticwp' ); ?>
                                    </h3>
                                    <p class="section-desc"><?php _e( 'The cost of the shipping service is prepaid by the customer.', 'anticwp' ); ?></p>
                                    
                                    <div class="input-row">
                                        <label class="input-label"><?php _e( 'Cost (€)', 'anticwp' ); ?></label>
                                        <input type="number" step="0.01" name="anticwp_shipping_cost" class="anticwp-input" value="<?php echo esc_attr( get_option( 'anticwp_shipping_cost', '5.00' ) ); ?>" />
                                    </div>
                                    <div class="input-row">
                                        <label class="input-label"><?php _e( 'Title at Checkout', 'anticwp' ); ?></label>
                                        <input type="text" name="anticwp_shipping_label" class="anticwp-input" value="<?php echo esc_attr( get_option( 'anticwp_shipping_label', __( 'Shipping Costs (Online Payment)', 'anticwp' ) ) ); ?>" />
                                        <p class="description" style="font-size: 11px; color: #666; margin-top: 5px;">
                                            <?php _e( 'This text appears on the shipping line at checkout.', 'anticwp' ); ?>
                                        </p>
                                    </div>
                                </div>

                                <div class="divider"></div>

                                <div class="section-block">
                                    <h3 class="section-title">
                                        <div class="icon-box green"><span class="dashicons dashicons-money-alt"></span></div>
                                        <?php _e( 'Cash on Delivery Cost', 'anticwp' ); ?>
                                    </h3>
                                    <p class="section-desc"><?php _e( 'The cost of the cash on delivery service is prepaid by the customer.', 'anticwp' ); ?></p>
                                    
                                    <div class="input-row">
                                        <label class="input-label"><?php _e( 'Cost (€)', 'anticwp' ); ?></label>
                                        <input type="number" step="0.01" name="anticwp_cod_cost" class="anticwp-input" value="<?php echo esc_attr( get_option( 'anticwp_cod_cost', '2.50' ) ); ?>" />
                                    </div>
                                    <div class="input-row">
                                        <label class="input-label"><?php _e( 'Title at Checkout', 'anticwp' ); ?></label>
                                        <input type="text" name="anticwp_cod_label" class="anticwp-input" value="<?php echo esc_attr( get_option( 'anticwp_cod_label', __( 'Cash on Delivery (Online Payment)', 'anticwp' ) ) ); ?>" />
                                    </div>
                                </div>

                                <div class="divider"></div>

                                <div class="section-block">
                                    <h3 class="section-title">
                                        <div class="icon-box" style="background:#fff7ed; color:#ea580c;"><span class="dashicons dashicons-performance"></span></div>
                                        <?php _e( 'Cache Compatibility', 'anticwp' ); ?>
                                    </h3>
                                    <p class="section-desc"><?php _e( 'Enable it if you use caching plugins (WP Rocket, LiteSpeed, Cloudflare).', 'anticwp' ); ?></p>
                                    
                                    <div class="input-row">
                                        <label class="input-label"><?php _e( 'Cache Compatibility Mode', 'anticwp' ); ?></label>
                                        <select name="anticwp_cache_compatible" class="anticwp-input" style="max-width: 200px;">
                                            <option value="yes" <?php selected($cache_compatible, 'yes'); ?>><?php _e( 'Active', 'anticwp' ); ?></option>
                                            <option value="no" <?php selected($cache_compatible, 'no'); ?>><?php _e( 'Inactive', 'anticwp' ); ?></option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="card-footer">
                                <button type="submit" class="save-btn">
                                    <span class="dashicons dashicons-saved"></span> <?php _e( 'Save Changes', 'anticwp' ); ?>
                                </button>
                            </div>
                        </div>
                    </form>
                </div>

                <div class="sidebar-column">
                    <div class="anticwp-card">
                        <div class="card-header">
                            <h2><span class="dashicons dashicons-shield header-icon"></span> <?php _e( 'License', 'anticwp' ); ?></h2>
                            <?php echo $this->license->get_status_label(); ?>
                        </div>
                        
                        <div class="card-body">
                            <form method="post" action="">
                                <?php wp_nonce_field( 'anticwp_license_action' ); ?>
                                <div class="license-key-wrapper">
                                    <span class="dashicons dashicons-admin-network"></span>
                                    <input type="text" name="license_key" 
                                           value="<?php echo esc_attr( get_option( 'anticwp_license_key' ) ); ?>" 
                                           class="anticwp-input license-input" 
                                           placeholder="ANTICWP-XXXX-XXXX">
                                </div>
                                <input type="submit" name="anticwp_activate_license" class="verify-btn" value="<?php _e( 'Activation & Control', 'anticwp' ); ?>">
                            </form>

                            <?php if ( $is_active ) : ?>
                                <div class="expiry-widget">
                                    <div class="widget-label"><?php _e( 'Remaining Time', 'anticwp' ); ?></div>
                                    <div class="widget-value">
                                        <?php if ($is_unlimited): ?>
                                            <span style="font-size: 40px; line-height: 1;">&infin;</span>
                                        <?php else: ?>
                                            <?php echo $days_left; ?>
                                        <?php endif; ?>
                                        <span class="widget-unit"><?php echo $is_unlimited ? __( 'Lifetime', 'anticwp' ) : __( 'Days', 'anticwp' ); ?></span>
                                    </div>
                                    <?php if (!$is_unlimited): ?>
                                        <div class="widget-date"><?php _e( 'Expiry:', 'anticwp' ); ?> <?php echo $expiry_display; ?></div>
                                    <?php endif; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
}